package gui.jtransitionsystem;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.FlowLayout;
import java.awt.Paint;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.Set;
import javax.swing.JButton;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JSplitPane;
import javax.swing.JTextArea;

import org.apache.commons.collections15.Transformer;
import org.apache.commons.collections15.functors.ChainedTransformer;

import edu.uci.ics.jung.algorithms.layout.FRLayout;
import edu.uci.ics.jung.graph.Graph;
import edu.uci.ics.jung.graph.SparseMultigraph;
import edu.uci.ics.jung.graph.util.EdgeType;
import edu.uci.ics.jung.visualization.VisualizationViewer;
import edu.uci.ics.jung.visualization.control.CrossoverScalingControl;
import edu.uci.ics.jung.visualization.control.DefaultModalGraphMouse;
import edu.uci.ics.jung.visualization.control.ScalingControl;
import edu.uci.ics.jung.visualization.control.ModalGraphMouse.Mode;
import edu.uci.ics.jung.visualization.decorators.ToStringLabeller;
import edu.uci.ics.jung.visualization.renderers.VertexLabelAsShapeRenderer;
import edu.uci.ics.jung.visualization.renderers.Renderer.VertexLabel.Position;

import service.SimpleTransition;
import service.SimpleState;
import service.TransitionSystem;

class TSSimpleActionEdge {
	private String label,from,to;
	public TSSimpleActionEdge(String label,String from,String to) {
		this.label=label;
		this.from=from;
		this.to=to;
	} 
	public String toString() { 
		return label;
	}
	public boolean equals(String label,String from,String to) { 
		return this.label.equals(label)&&this.from.equals(from)&&this.to.equals(to);
	}
}

public class JTransitionSystemSimple {
	private Graph<String,TSSimpleActionEdge> graph= new SparseMultigraph<String,TSSimpleActionEdge>();
	private TransitionSystem<SimpleState, SimpleTransition> ts;
	private VisualizationViewer<String,TSSimpleActionEdge> vv;
	private SimpleState currentState=null;
	private SimpleTransition lastAction=null;
	private JPanel panelTransitionSystem;

	@SuppressWarnings("unchecked")
	public JTransitionSystemSimple(TransitionSystem<SimpleState, SimpleTransition> ts) {
		
		
		FRLayout<String, TSSimpleActionEdge> layout = new FRLayout(graph);
		layout.setRepulsionMultiplier(200);
		layout.setAttractionMultiplier(2);
		vv = new VisualizationViewer<String,TSSimpleActionEdge>(layout);
    	 
    	this.ts=ts;
		this.currentState=ts.getInitial();
		addStates();
		addActions();

		
		// Setup up a new vertex to paint transformer...
		Transformer<String,Paint> vertexPaint = new Transformer<String,Paint>() {
			public Paint transform(String i) {
				if(i.equals(currentState.getName()))
					return Color.GREEN;
				else
					return Color.CYAN;
			}
			};
			Transformer<TSSimpleActionEdge,Paint> edgePaint = new Transformer<TSSimpleActionEdge,Paint>() {
				public Paint transform(TSSimpleActionEdge actionEdge) {
						if (lastAction!= null && actionEdge.equals(lastAction.getName(),
								              lastAction.getStateFrom().getName(),
								              lastAction.getStateTo().getName()
								             )
					       )
							return Color.RED;
						else return Color.BLACK;
				}
				};
		vv.getRenderContext().setVertexFillPaintTransformer(vertexPaint);
		vv.getRenderContext().setVertexLabelTransformer(new ToStringLabeller<String>());
		vv.getRenderContext().setEdgeLabelTransformer(new ToStringLabeller<TSSimpleActionEdge>());
		vv.getRenderer().getVertexLabelRenderer().setPosition(Position.CNTR);
		vv.getRenderContext().setEdgeDrawPaintTransformer(edgePaint);
        VertexLabelAsShapeRenderer<String,TSSimpleActionEdge> vlasr = new VertexLabelAsShapeRenderer<String,TSSimpleActionEdge>(vv.getRenderContext());
        vv.getRenderer().setVertexLabelRenderer(vlasr);
        vv.getRenderContext().setVertexShapeTransformer(vlasr);
        vv.getRenderContext().setVertexLabelTransformer(
        		// this chains together Transformers so that the html tags
        		// are prepended to the toString method output
        		new ChainedTransformer<String,String>(new Transformer[]{
        		new ToStringLabeller<String>(),
        		new Transformer<String,String>() {
					public String transform(String input) {
						return "<html><center>"+input;
					}}}));
        vv.getRenderContext().setEdgeLabelTransformer(
        		// this chains together Transformers so that the html tags
        		// are prepended to the toString method output
        		new ChainedTransformer<TSSimpleActionEdge,String>(new Transformer[]{
        		new ToStringLabeller<String>(),
        		new Transformer<String,String>() {
					public String transform(String input) {
						return "<html><center><font size=\"4\" color=\"#ff0000\">"+input;
					}}}));
        
		DefaultModalGraphMouse<String, TSComposedActionEdge> gm = new DefaultModalGraphMouse<String, TSComposedActionEdge>();
		gm.setMode(Mode.EDITING);
		vv.setGraphMouse(gm);
		

		JPanel panelButtons=new JPanel(new FlowLayout());

		final ScalingControl scaler = new CrossoverScalingControl();	
        JButton zoomIn = new JButton("Zoom in");
        zoomIn.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                scaler.scale(vv, 1.1f, vv.getCenter());
            }
        });
        panelButtons.add(zoomIn);

        JButton zoomOut = new JButton("Zoom out");
        zoomOut.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                scaler.scale(vv, 1/1.1f, vv.getCenter());
            }
        });
        panelButtons.add(zoomOut);

        
        JButton editingMode = new JButton("Translation Mode");
        editingMode.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
            	setModeEditing();
            }
        });
        panelButtons.add(editingMode);

        JButton pickingMode = new JButton("Picking mode");
        pickingMode.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
            	setModePicking();
            }
        });
        panelButtons.add(pickingMode);

		JPanel panelGraph=new JPanel();
		panelGraph.add(vv);

		JTextArea textDotArea=new JTextArea(getDotFormatString());
		textDotArea.setEditable(false);
		JScrollPane scroolPaneDotArea=new JScrollPane(textDotArea);
		JSplitPane splitPane=new JSplitPane(JSplitPane.VERTICAL_SPLIT,panelGraph,scroolPaneDotArea);
		splitPane.setDividerLocation(600);
		
		panelTransitionSystem=new JPanel(new BorderLayout());


		
		panelTransitionSystem.add(panelButtons,BorderLayout.NORTH);
		panelTransitionSystem.add(splitPane,BorderLayout.CENTER);
	}
	
	private String getDotFormatString(){
		StringBuffer result=new StringBuffer();
		result.append("digraph "+ts.getName()+ " {\n");
		Iterator<SimpleTransition> itSimpleAction=ts.getAllActions().iterator();
		while(itSimpleAction.hasNext()){
			SimpleTransition currentAction=itSimpleAction.next();
			String currentActionName=currentAction.getName();
			String currentSourceName=currentAction.getStateFrom().getName();
			String currentTargetName=currentAction.getStateTo().getName();
			result.append("\t"+currentSourceName+" -> "+currentTargetName+" [label=\""+currentActionName+"\"]\n");
		}
		
		LinkedList<String> initialStatesStr=new LinkedList<String>();
		LinkedList<String> finalStatesStr=new LinkedList<String>();
		Iterator<SimpleState> itAllStates=ts.getAllStates().iterator();
		while(itAllStates.hasNext()){
			SimpleState currentState=itAllStates.next();
			if(currentState.isInitial())
				initialStatesStr.add(currentState.getName());
			if(currentState.isFinal())
				finalStatesStr.add(currentState.getName());
		}
		
		result.append("\t[initial = {");
		Iterator<String> itInitial=initialStatesStr.iterator();
		boolean firstInitialAdded=false;
		while(itInitial.hasNext()){
			if(!firstInitialAdded){
				firstInitialAdded=true;
				result.append(itInitial.next());
			}else
				result.append(","+itInitial.next());
		}
		result.append("} ]\n");

		result.append("\t[final = {");
		Iterator<String> itFinal=finalStatesStr.iterator();
		boolean firstFinalAdded=false;
		while(itFinal.hasNext()){
			if(!firstFinalAdded){
				firstFinalAdded=true;
				result.append(itFinal.next());
			}else
				result.append(","+itFinal.next());
		}
		result.append("} ]\n");
		
		
		result.append("}\n");
		return result.toString();
	}

	public void setModeEditing(){
		DefaultModalGraphMouse<String, TSComposedActionEdge> gm = new DefaultModalGraphMouse<String, TSComposedActionEdge>();
		gm.setMode(Mode.EDITING);
		vv.setGraphMouse(gm);
	}
	public void setModePicking(){
		DefaultModalGraphMouse<String, TSComposedActionEdge> gm = new DefaultModalGraphMouse<String, TSComposedActionEdge>();
		gm.setMode(Mode.PICKING);
		vv.setGraphMouse(gm);
	}
	public SimpleState getCurrentState(){
		return currentState;
	}

	public JPanel getVisualizationViewer(){
		return panelTransitionSystem;
	}

	private void addStates(){
		Iterator<SimpleState> itState=(Iterator<SimpleState>)ts.getAllStates().iterator();
		while(itState.hasNext()){
			SimpleState state=itState.next();
			graph.addVertex(state.getName());
		}
	}
	private void addActions(){
		Iterator<SimpleTransition> itAction=(Iterator<SimpleTransition>)ts.getAllActions().iterator();
		while(itAction.hasNext()){
			SimpleTransition action=itAction.next();
			graph.addEdge(new TSSimpleActionEdge(action.getName(),action.getStateFrom().getName(),action.getStateTo().getName()),
					      action.getStateFrom().getName(),
					      action.getStateTo().getName(),EdgeType.DIRECTED);
		}
		
	}
	public Set<SimpleTransition> getPossibleAction() throws Exception{
		return ts.getActionsOf(currentState);
	}
	public void executeAction(SimpleTransition action) throws Exception{
		lastAction=action;
		currentState=(SimpleState)action.getStateTo();
	}
}
